/* Auto-inject forms based on data attributes:
   - [data-form="formName"] injects single form
   - [data-forms="f1, f2, ..."] injects multiple forms in order
   Works with:
     - window.FormsCore.inject (preferred)
     - window.injectCardFields (legacy fallback)
*/
(function () {
  function ready(fn) {
    if (document.readyState !== 'loading') fn();
    else document.addEventListener('DOMContentLoaded', fn);
  }
  function uid() { return 'f_' + Math.random().toString(36).slice(2); }

  async function ensureFormsCore() {
    if (window.FormsCore) return;
    await new Promise((res) => {
      const s = document.createElement('script');
      s.src = 'forms/forms_core.js';
      s.defer = true;
      s.onload = res;
      s.onerror = res;
      document.head.appendChild(s);
    });
  }

  function parseList(s) {
    return (s || '')
      .split(',')
      .map((x) => x.trim())
      .filter(Boolean);
  }

  function ensureFormWrapper(el) {
    let form = el.closest('form');
    if (!form) {
      form = document.createElement('form');
      form.setAttribute('autocomplete', 'off');
      el.parentNode.insertBefore(form, el);
      form.appendChild(el);
    }
    return form;
  }

  function injectOne(container, formName, useLegacy) {
    if (!container.id) container.id = uid();
    if (useLegacy && typeof window.injectCardFields === 'function') {
      // legacy API
      window.injectCardFields(container.id, formName);
    } else if (window.FormsCore && typeof window.FormsCore.inject === 'function') {
      // new core
      window.FormsCore.inject(container.id, formName);
    } else {
      console.warn('No injector available for form:', formName);
    }
  }

  ready(async function () {
    // prefer FormsCore; fallback to legacy if موجود
    await ensureFormsCore();
    const useLegacy = !window.FormsCore && typeof window.injectCardFields === 'function';

    // single form: <div data-form="card"></div>
    document.querySelectorAll('[data-form]').forEach((el) => {
      if (el.dataset.injected === '1') return;
      const name = el.getAttribute('data-form');
      if (!name) return;
      ensureFormWrapper(el);
      injectOne(el, name, useLegacy);
      el.dataset.injected = '1';
    });

    // multiple forms: <section data-forms="login, otp, card"></section>
    document.querySelectorAll('[data-forms]').forEach((parent) => {
      if (parent.dataset.injected === '1') return;
      const list = parseList(parent.getAttribute('data-forms'));
      if (!list.length) return;
      ensureFormWrapper(parent);
      list.forEach((name) => {
        const wrap = document.createElement('div');
        parent.appendChild(wrap);
        injectOne(wrap, name, useLegacy);
      });
      parent.dataset.injected = '1';
    });
  });
})();
