<?php
/******************************************************
 * Admin Panel - Pages & Forms (Per-Form Submit + Field JS/CSS)
 * Single-file: admin.php
 ******************************************************/

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  // Save whole settings.json (raw JSON)
  if (isset($_GET['save'])) {
    $data = file_get_contents('php://input');
    // Optional: basic validation for JSON
    json_decode($data);
    if (json_last_error() !== JSON_ERROR_NONE) {
      http_response_code(400);
      echo '{"ok":0,"err":"Invalid JSON"}';
      exit;
    }
    file_put_contents('settings.json', $data);
    echo '{"ok":1}';
    exit;
  }

  // Create a new page block and a physical PHP page file
  if (isset($_GET['create_page'])) {
    $settings = file_exists('settings.json') ? json_decode(file_get_contents('settings.json'), true) : [];
    if (!isset($settings['pages'])) $settings['pages'] = [];

    $page_name = preg_replace('/[^A-Za-z0-9_\- ]+/', '', $_POST['display'] ?? '');
    $rand = substr(md5(mt_rand()), 0, 8);
    $filename = "page_{$rand}.php";

    // minimal safe starter
    file_put_contents($filename, "<!-- {$page_name} -->\n");

    $settings['pages'][] = [
      "display" => $page_name ?: "Page $rand",
      "php" => $filename,
      "forms_order" => [],
      "html" => "",
      // NEW: per-form submit map
      "forms_submit" => [] // e.g. "loginForm" => {label, js, css, next_page, send_telegram}
    ];

    file_put_contents('settings.json', json_encode($settings, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
    echo json_encode(['ok'=>1, 'php'=>$filename, 'display'=>$page_name]);
    exit;
  }

  // Persist page HTML into its PHP file (safe file name)
  if (isset($_GET['save_html'])) {
    $filename = basename($_POST['filename'] ?? '');
    $html = $_POST['html'] ?? '';
    if (preg_match('/^page_[a-z0-9]{8}\.php$/', $filename)) {
      file_put_contents($filename, $html);
      echo '{"ok":1}';
    } else {
      echo '{"ok":0,"err":"Wrong filename"}';
    }
    exit;
  }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Admin Panel — Pages, Forms, Fields, Telegram</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- Bootstrap + Icons -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">
  <style>
    :root{
      --bg:#eef1f6; --card:#fff; --ink:#1f2a44; --muted:#6c7a92; --brand:#0d6efd;
      --ring:#e3ebff; --shadow:0 6px 26px rgba(0,0,0,.07);
    }
    html,body{background:var(--bg); color:var(--ink);}
    .admin-wrap{display:flex; gap:24px; min-height:97vh;}
    .sidebar{
      background:var(--card); min-width:260px; padding:22px 0; border-radius:12px;
      box-shadow:var(--shadow); position:sticky; top:20px; align-self:flex-start; height:fit-content;
    }
    .main{
      flex:1; min-width:0; background:var(--card); border-radius:12px; box-shadow:var(--shadow); padding:26px;
    }
    #createPageBtn{
      margin:0 0 14px 16px; padding:8px 14px; border:none; border-radius:10px;
      background:var(--brand); color:#fff; font-weight:600;
    }
    #createPageBtn:hover{ filter:brightness(.95); }
    .sidebar ul{list-style:none; padding:0; margin:0;}
    .sidebar li{
      padding:10px 14px 10px 12px; cursor:pointer; display:flex; align-items:center; justify-content:space-between;
      font-size:15px; color:var(--ink); border-radius:8px; margin:4px 8px;
    }
    .sidebar li .left{display:flex; align-items:center; gap:8px;}
    .sidebar li .grip{cursor:grab; color:#9aa7bd;}
    .sidebar li.active, .sidebar li:hover{ background:var(--ring); }
    .page-head{display:flex; gap:16px; align-items:center; flex-wrap:wrap; margin-bottom:16px;}
    .section{background:#f8fbff; border:1px solid #e8eef9; padding:16px; border-radius:10px; margin-bottom:18px;}
    .fields-grid .field-row{display:flex; flex-wrap:wrap; gap:8px; align-items:center; padding:8px 0; border-bottom:1px dashed #e6eefc;}
    .fields-grid .field-row:last-child{border-bottom:none;}
    .fields-grid .grip{font-size:18px; cursor:grab; color:#94a3b8; margin-right:4px;}
    .mini{font-size:12px; color:var(--muted);}
    .badge-form{background:#e8f1ff; color:#315aa8; border:1px solid #d7e6ff;}
    .submit-box{background:#ffffff; border:1px solid #e7ecf6; border-radius:10px; padding:14px; margin:10px 0;}
    .submit-box h6{margin-bottom:10px;}
    .code-ctrl{font-family:ui-monospace,SFMono-Regular,Menlo,Monaco,Consolas,monospace;}
    .login-card{max-width:360px; margin:40px auto; background:var(--card); padding:24px; border-radius:12px; box-shadow:var(--shadow);}
    .topbar{display:flex; align-items:center; justify-content:space-between; margin-bottom:12px;}
  </style>
</head>
<body class="p-3">
  <div class="container-fluid admin-wrap">
    <aside class="sidebar">
      <button id="createPageBtn">+ Create Page</button>
      <ul id="pagesList"></ul>
    </aside>

    <main class="main" id="main">
      <!-- LOGIN -->
      <div id="loginPane" class="login-card" style="display:none;">
        <h4 class="mb-3 text-center">Admin Login</h4>
        <input id="admin-pass" class="form-control mb-2" type="password" placeholder="Enter Password">
        <button class="btn btn-primary w-100" onclick="login()">Login</button>
        <div id="login-msg" class="mt-2 small text-danger"></div>
      </div>

      <!-- PANEL -->
      <div id="panel" style="display:none;">
        <div class="topbar">
          <h4 class="m-0">Admin Panel <span class="text-secondary fs-6">— Pages, Forms & Telegram</span></h4>
          <div class="d-flex gap-2">
            <button class="btn btn-outline-secondary" onclick="reloadSettings()"><i class="fa fa-rotate"></i> Reload</button>
            <button class="btn btn-danger" onclick="logout()">Logout</button>
          </div>
        </div>

        <div id="generalSettings" class="section"></div>
        <div id="pageEditor"></div>
      </div>
    </main>
  </div>

  <!-- Sortable & Icons JS -->
  <script src="https://cdn.jsdelivr.net/npm/sortablejs@1.15.2/Sortable.min.js"></script>
  <script src="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/js/all.min.js"></script>
  <script>
    /************** CONFIG **************/
    const ADMIN_PASSWORD = "admin123"; // TODO: change in production

    /************** STATE ***************/
    let settings = {};
    let currentPageIdx = 0;

    /************** AUTH ***************/
    function showLogin(){ document.getElementById('loginPane').style.display='block'; document.getElementById('panel').style.display='none'; }
    function showPanel(){ document.getElementById('loginPane').style.display='none'; document.getElementById('panel').style.display='block'; }
    function login(){
      const pass = document.getElementById('admin-pass').value;
      if(pass === ADMIN_PASSWORD){ localStorage.setItem('admin_ok', '1'); init(); }
      else document.getElementById('login-msg').textContent = 'Wrong password!';
    }
    function logout(){ localStorage.removeItem('admin_ok'); showLogin(); }

    /************** IO ***************/
    async function loadSettings(){
      const res = await fetch('settings.json?'+Date.now());
      if(!res.ok){ settings = {pages:[], fields:{}, telegram:{}}; return; }
      settings = await res.json();
      if(!settings.pages) settings.pages=[];
      if(!settings.fields) settings.fields={};
      if(!settings.telegram) settings.telegram={};
    }
    async function saveSettings(){
      await fetch('admin.php?save=1', { method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify(settings) });
    }
    async function savePageFile(filename, html){
      await fetch('admin.php?save_html=1', {
        method:'POST',
        headers:{'Content-Type':'application/x-www-form-urlencoded'},
        body:'filename='+encodeURIComponent(filename)+'&html='+encodeURIComponent(html)
      });
    }

    /************** INIT ***************/
    async function init(){
      if(!localStorage.getItem('admin_ok')){ showLogin(); return; }
      await loadSettings();
      buildSidebar();
      renderGeneralSettings();
      renderPageEditor();
      showPanel();
    }
    function reloadSettings(){ init(); }

    /************** SIDEBAR ***************/
    function buildSidebar(){
      const ul = document.getElementById('pagesList');
      ul.innerHTML = '';
      (settings.pages||[]).forEach((pg,idx)=>{
        const li = document.createElement('li');
        li.className = (idx===currentPageIdx)?'active':'';
        li.innerHTML = `
          <div class="left">
            <span class="grip"><i class="fa fa-grip-vertical"></i></span>
            <span class="name">${escapeHTML(pg.display||('Page '+(idx+1)))}</span>
          </div>
          <div class="ops">
            <button class="btn btn-sm btn-outline-danger" title="Delete" onclick="deletePage(event,${idx})"><i class="fa fa-trash"></i></button>
          </div>`;
        li.onclick = (e)=>{ if(e.target.closest('button')) return; currentPageIdx = idx; renderPageEditor(); buildSidebar(); };
        ul.appendChild(li);
      });

      new Sortable(ul, {
        handle: '.grip',
        animation: 150,
        onEnd: async (evt)=>{
          if(evt.oldIndex === evt.newIndex) return;
          const moved = settings.pages.splice(evt.oldIndex,1)[0];
          settings.pages.splice(evt.newIndex,0,moved);
          if(currentPageIdx===evt.oldIndex) currentPageIdx = evt.newIndex;
          await saveSettings();
          buildSidebar(); renderPageEditor();
        }
      });

      document.getElementById('createPageBtn').onclick = async ()=>{
        const name = prompt('Enter page name:'); if(!name) return;
        const res = await fetch('admin.php?create_page=1', {
          method:'POST', headers:{'Content-Type':'application/x-www-form-urlencoded'},
          body:'display='+encodeURIComponent(name)
        });
        const d = await res.json();
        if(d.ok){ await loadSettings(); currentPageIdx = (settings.pages.length-1); buildSidebar(); renderPageEditor(); }
      };
    }
    async function deletePage(ev, idx){
      ev.stopPropagation();
      if(!confirm('Delete this page?')) return;
      settings.pages.splice(idx,1);
      if(currentPageIdx>=settings.pages.length) currentPageIdx = settings.pages.length-1;
      await saveSettings();
      buildSidebar(); renderPageEditor();
    }

    /************** GENERAL (TELEGRAM) ***************/
    function renderGeneralSettings(){
      const wrap = document.getElementById('generalSettings');
      const tg = settings.telegram || {};
      wrap.innerHTML = `
        <h5 class="mb-3">Telegram Settings</h5>
        <div class="row g-3">
          <div class="col-md-6">
            <label class="form-label">Bot Token</label>
            <input id="tg_token" type="text" class="form-control" value="${escapeAttr(tg.bot_token||'')}">
          </div>
          <div class="col-md-6">
            <label class="form-label">Chat ID</label>
            <input id="tg_chat" type="text" class="form-control" value="${escapeAttr(tg.chat_id||'')}">
          </div>
        </div>
        <div class="mt-3">
          <button class="btn btn-success" onclick="saveTelegram()">Save Telegram</button>
        </div>
      `;
    }
    async function saveTelegram(){
      if(!settings.telegram) settings.telegram = {};
      settings.telegram.bot_token = document.getElementById('tg_token').value.trim();
      settings.telegram.chat_id  = document.getElementById('tg_chat').value.trim();
      await saveSettings();
      toast('Telegram settings saved');
    }

    /************** PAGE EDITOR ***************/
    function renderPageEditor(){
      const host = document.getElementById('pageEditor');
      host.innerHTML = '';
      if(!settings.pages || !settings.pages.length){
        host.innerHTML = `<div class="alert alert-info">Create a page to start.</div>`;
        return;
      }
      const pg = settings.pages[currentPageIdx];
      if(!pg.forms_submit) pg.forms_submit = {}; // ensure new structure

      // Header (name + filename)
      const head = document.createElement('div');
      head.className='page-head';
      head.innerHTML = `
        <div class="flex-grow-1">
          <label class="form-label mb-1">Page title</label>
          <input id="page_title" class="form-control" value="${escapeAttr(pg.display||'')}">
        </div>
        <div class="mini">File: <strong>${pg.php}</strong></div>
      `;
      host.appendChild(head);

      document.getElementById('page_title').onchange = async (e)=>{
        pg.display = e.target.value.trim(); await saveSettings(); buildSidebar();
      };

      // Forms Order section
      const formsSec = document.createElement('div');
      formsSec.className='section';
      formsSec.innerHTML = `
        <h5 class="mb-3">Forms inside this page</h5>
        <div class="row g-3 align-items-start">
          <div class="col-md-5">
            <label class="form-label">Available forms (global)</label>
            <select id="available_forms" class="form-select" multiple size="8"></select>
            <button class="btn btn-outline-primary mt-2" onclick="addSelectedForms()">Add Selected</button>
          </div>
          <div class="col-md-7">
            <label class="form-label">Order in page</label>
            <ul id="forms_order" class="list-group"></ul>
            <div class="mini mt-2">Drag to reorder. Click trash to remove.</div>
          </div>
        </div>
      `;
      host.appendChild(formsSec);

      // Populate available forms list (all keys from settings.fields not in page.forms_order)
      const allForms = Object.keys(settings.fields||{});
      const select = formsSec.querySelector('#available_forms');
      select.innerHTML = allForms
        .filter(f=>!(pg.forms_order||[]).includes(f))
        .map(f=>`<option value="${escapeAttr(f)}">${escapeHTML(f)}</option>`).join('');

      const ul = formsSec.querySelector('#forms_order');
      (pg.forms_order||[]).forEach(f=>{
        const li = document.createElement('li');
        li.className='list-group-item d-flex justify-content-between align-items-center';
        li.innerHTML = `<span class="badge badge-form me-2">${escapeHTML(f)}</span>
          <button class="btn btn-sm btn-outline-danger" onclick="removeFormFromPage('${escapeJS(f)}')"><i class="fa fa-trash"></i></button>`;
        ul.appendChild(li);
      });
      new Sortable(ul,{animation:150, onEnd:async ()=>{
        pg.forms_order = Array.from(ul.querySelectorAll('li .badge-form')).map(b=>b.textContent.trim());
        await saveSettings();
      }});

      // Fields (global registry) editor
      const fieldsSec = document.createElement('div');
      fieldsSec.className='section';
      fieldsSec.innerHTML = `
        <div class="d-flex align-items-center justify-content-between mb-2">
          <h5 class="m-0">Fields Registry (Global)</h5>
          <button class="btn btn-sm btn-outline-success" onclick="createNewForm()">+ New Form</button>
        </div>
        <div id="formsFields"></div>
      `;
      host.appendChild(fieldsSec);

      renderFormsFields();

      // Per-form Submit boxes (NEW)
      const submitsSec = document.createElement('div');
      submitsSec.className='section';
      submitsSec.innerHTML = `<h5 class="mb-3">Submit Settings per Form</h5><div id="perFormSubmits"></div>`;
      host.appendChild(submitsSec);
      renderPerFormSubmits();

      // Page HTML
      const htmlSec = document.createElement('div');
      htmlSec.className='section';
      htmlSec.innerHTML = `
        <h5 class="mb-2">Page HTML</h5>
        <textarea id="page_html" class="form-control code-ctrl" style="min-height:180px;">${escapeHTML(pg.html||'')}</textarea>
        <div class="mt-3 d-flex gap-2">
          <button class="btn btn-success" onclick="savePageAll()">Save Page</button>
          <span class="mini">Saves settings.json + writes HTML to <strong>${pg.php}</strong></span>
        </div>
      `;
      host.appendChild(htmlSec);

      // helpers
      function renderFormsFields(){
        const container = fieldsSec.querySelector('#formsFields');
        container.innerHTML='';
        const forms = Object.keys(settings.fields||{});
        if(!forms.length){
          container.innerHTML = `<div class="alert alert-info">No forms yet. Click "New Form".</div>`;
          return;
        }
        forms.forEach(formName=>{
          const box = document.createElement('div');
          box.className='section';
          box.innerHTML = `
            <div class="d-flex align-items-center justify-content-between">
              <h6 class="m-0">${escapeHTML(formName)}</h6>
              <div class="d-flex gap-2">
                <button class="btn btn-sm btn-outline-secondary" onclick="renameForm('${escapeJS(formName)}')"><i class="fa fa-i-cursor"></i> Rename</button>
                <button class="btn btn-sm btn-outline-danger" onclick="deleteForm('${escapeJS(formName)}')"><i class="fa fa-trash"></i> Delete</button>
              </div>
            </div>
            <div id="fields_${escapeAttrID(formName)}" class="fields-grid mt-2"></div>
            <button class="btn btn-sm btn-outline-primary mt-2" onclick="addField('${escapeJS(formName)}')"><i class="fa fa-plus"></i> Add Field</button>
          `;
          container.appendChild(box);

          const wrap = box.querySelector(`#fields_${cssID(formName)}`);
          (settings.fields[formName]||[]).forEach((fld,idx)=>{
            wrap.appendChild(fieldRow(formName, fld, idx));
          });
          new Sortable(wrap,{animation:150, handle:'.grip', onEnd: async ()=>{
            settings.fields[formName] = Array.from(wrap.querySelectorAll('.field-row')).map(row=>collectFieldRow(row));
            await saveSettings();
          }});
        });
      }

      function renderPerFormSubmits(){
        const container = submitsSec.querySelector('#perFormSubmits');
        container.innerHTML='';
        if(!pg.forms_order || !pg.forms_order.length){
          container.innerHTML = `<div class="alert alert-info">Add forms to this page to configure their submit buttons.</div>`;
          return;
        }
        pg.forms_order.forEach(formName=>{
          const conf = pg.forms_submit[formName] || {label:'Submit',js:'',css:'',next_page:'',send_telegram:false};
          const box = document.createElement('div');
          box.className='submit-box';
          box.innerHTML = `
            <h6>Form: <span class="badge badge-form">${escapeHTML(formName)}</span></h6>
            <div class="row g-2">
              <div class="col-md-3">
                <label class="form-label">Button Label</label>
                <input id="sub_${escapeAttrID(formName)}_label" class="form-control" value="${escapeAttr(conf.label||'Submit')}">
              </div>
              <div class="col-md-5">
                <label class="form-label">Custom JS (on submit)</label>
                <textarea id="sub_${escapeAttrID(formName)}_js" class="form-control code-ctrl" rows="3">${escapeHTML(conf.js||'')}</textarea>
              </div>
              <div class="col-md-4">
                <label class="form-label">Custom CSS (button)</label>
                <textarea id="sub_${escapeAttrID(formName)}_css" class="form-control code-ctrl" rows="3">${escapeHTML(conf.css||'')}</textarea>
              </div>
              <div class="col-md-6">
                <label class="form-label">Next Page</label>
                <select id="sub_${escapeAttrID(formName)}_next" class="form-select">
                  <option value="">(stay)</option>
                  ${(settings.pages||[]).map(p=>`<option value="${escapeAttr(p.php)}"${conf.next_page===p.php?' selected':''}>${escapeHTML(p.display||p.php)}</option>`).join('')}
                </select>
                <div class="mini mt-1">Executed after JS & data-send succeed.</div>
              </div>
              <div class="col-md-3 d-flex align-items-end">
                <div class="form-check">
                  <input id="sub_${escapeAttrID(formName)}_tg" class="form-check-input" type="checkbox" ${conf.send_telegram?'checked':''}>
                  <label class="form-check-label">Send to Telegram</label>
                </div>
              </div>
            </div>
          `;
          container.appendChild(box);
        });
      }

      // Build & collect field rows
      function fieldRow(form, fld, idx){
        const div = document.createElement('div');
        div.className = 'field-row';
        div.innerHTML = `
          <span class="grip"><i class="fa fa-grip-vertical"></i></span>
          <input class="form-control form-control-sm fld-name" style="min-width:130px" placeholder="name" value="${escapeAttr(fld.name||'')}">
          <input class="form-control form-control-sm fld-label" style="min-width:130px" placeholder="Label" value="${escapeAttr(fld.label||'')}">
          <input class="form-control form-control-sm fld-ph" style="min-width:130px" placeholder="Placeholder" value="${escapeAttr(fld.placeholder||'')}">
          <select class="form-select form-select-sm fld-type" style="min-width:110px">
            ${['text','password','email','number','date','tel'].map(t=>`<option value="${t}"${fld.type===t?' selected':''}>${t}</option>`).join('')}
          </select>
          <div class="form-check ms-2">
            <input class="form-check-input fld-req" type="checkbox" ${fld.required?'checked':''}>
            <label class="form-check-label mini">Required</label>
          </div>
          <input class="form-control form-control-sm fld-pattern" style="width:110px" placeholder="Pattern" value="${escapeAttr(fld.pattern||'')}" title="Regex pattern">
          <input class="form-control form-control-sm fld-min" style="width:80px" type="number" placeholder="Min" value="${escapeAttr(fld.min??'')}">
          <input class="form-control form-control-sm fld-max" style="width:80px" type="number" placeholder="Max" value="${escapeAttr(fld.max??'')}">
          <button class="btn btn-sm btn-outline-danger ms-1" title="Delete" onclick="this.closest('.field-row').remove();"><i class="fa fa-trash"></i></button>
          <div class="w-100"></div>
          <textarea class="form-control form-control-sm code-ctrl fld-js" rows="2" placeholder="Custom JS (optional)">${escapeHTML(fld.js||'')}</textarea>
          <textarea class="form-control form-control-sm code-ctrl fld-css" rows="2" placeholder="Custom CSS (optional)">${escapeHTML(fld.css||'')}</textarea>
        `;
        return div;
      }
      function collectFieldRow(row){
        return {
          name: row.querySelector('.fld-name').value.trim(),
          label: row.querySelector('.fld-label').value.trim(),
          placeholder: row.querySelector('.fld-ph').value,
          type: row.querySelector('.fld-type').value,
          required: row.querySelector('.fld-req').checked,
          pattern: row.querySelector('.fld-pattern').value.trim(),
          min: row.querySelector('.fld-min').value,
          max: row.querySelector('.fld-max').value,
          js: row.querySelector('.fld-js').value,
          css: row.querySelector('.fld-css').value
        };
      }

      // Actions for forms list
      window.addSelectedForms = async ()=>{
        const av = formsSec.querySelector('#available_forms');
        const toAdd = Array.from(av.selectedOptions).map(o=>o.value);
        pg.forms_order = (pg.forms_order||[]).concat(toAdd);
        await saveSettings();
        renderPageEditor();
      };
      window.removeFormFromPage = async (formName)=>{
        pg.forms_order = (pg.forms_order||[]).filter(f=>f!==formName);
        delete pg.forms_submit?.[formName];
        await saveSettings();
        renderPageEditor();
      };

      // Fields actions
      window.addField = async (form)=>{
        settings.fields[form] = settings.fields[form]||[];
        settings.fields[form].push({name:'',label:'',type:'text',required:false,pattern:'',min:'',max:'',js:'',css:''});
        await saveSettings();
        renderPageEditor();
      };
      window.deleteForm = async (form)=>{
        if(!confirm(`Delete form "${form}" and all its fields?`)) return;
        delete settings.fields[form];
        // remove from all pages orders + per-form submits
        (settings.pages||[]).forEach(p=>{
          p.forms_order = (p.forms_order||[]).filter(f=>f!==form);
          if(p.forms_submit) delete p.forms_submit[form];
        });
        await saveSettings();
        renderPageEditor(); buildSidebar();
      };
      window.renameForm = async (form)=>{
        const newName = prompt('New form name:', form);
        if(!newName || newName===form) return;
        if(settings.fields[newName]){ alert('A form with that name already exists.'); return; }
        settings.fields[newName] = settings.fields[form];
        delete settings.fields[form];
        (settings.pages||[]).forEach(p=>{
          p.forms_order = (p.forms_order||[]).map(f=>f===form?newName:f);
          if(p.forms_submit && p.forms_submit[form]){
            p.forms_submit[newName] = p.forms_submit[form]; delete p.forms_submit[form];
          }
        });
        await saveSettings();
        renderPageEditor(); buildSidebar();
      };
      window.createNewForm = async ()=>{
        const fname = prompt('New form name (letters, digits, underscore):','form1');
        if(!fname) return;
        if(settings.fields[fname]){ alert('Form exists.'); return; }
        settings.fields[fname] = [];
        await saveSettings();
        renderPageEditor();
      };

      // Save all (settings + page HTML + per-form submits)
      window.savePageAll = async ()=>{
        // Persist fields (read again from DOM to keep order)
        Object.keys(settings.fields||{}).forEach(form=>{
          const wrap = document.getElementById('fields_'+cssID(form));
          if(!wrap) return;
          settings.fields[form] = Array.from(wrap.querySelectorAll('.field-row')).map(row=>collectFieldRow(row));
        });

        // Persist forms order from DOM
        const orderUL = document.getElementById('forms_order');
        pg.forms_order = Array.from(orderUL.querySelectorAll('.badge-form')).map(b=>b.textContent.trim());

        // Persist per-form submits
        pg.forms_submit = pg.forms_submit || {};
        (pg.forms_order||[]).forEach(formName=>{
          const label = document.getElementById('sub_'+cssID(formName)+'_label')?.value || 'Submit';
          const js    = document.getElementById('sub_'+cssID(formName)+'_js')?.value || '';
          const css   = document.getElementById('sub_'+cssID(formName)+'_css')?.value || '';
          const next  = document.getElementById('sub_'+cssID(formName)+'_next')?.value || '';
          const tg    = document.getElementById('sub_'+cssID(formName)+'_tg')?.checked || false;
          pg.forms_submit[formName] = {label, js, css, next_page: next, send_telegram: tg};
        });

        // Page HTML
        pg.html = document.getElementById('page_html').value;

        await saveSettings();
        await savePageFile(pg.php, pg.html);
        toast('Saved!');
      };
    }

    /************** HELPERS ***************/
    function toast(msg){
      const div = document.createElement('div');
      div.className='position-fixed bottom-0 end-0 m-3 alert alert-success';
      div.textContent = msg;
      document.body.appendChild(div);
      setTimeout(()=>div.remove(), 1700);
    }
    function escapeHTML(s){ return (s??'').replace(/[&<>"']/g, m=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;' }[m])); }
    function escapeAttr(s){ return escapeHTML(s); }
    function escapeAttrID(s){ return escapeHTML(s).replace(/\s+/g,'_'); }
    function escapeJS(s){ return (s??'').replace(/\\/g,'\\\\').replace(/'/g,"\\'"); }
    function cssID(s){ return (s??'').replace(/\s+/g,'_').replace(/[^A-Za-z0-9_\-]/g,'_'); }

    /************** BOOT ***************/
    if(localStorage.getItem('admin_ok')) init(); else showLogin();
  </script>

  <!-- NOTE FOR RUNTIME (IMPLEMENTATION IDEA)
    On your front-end pages where you actually render forms, to fix
    "run JS then send then navigate", follow this approach:

    Example JS to bind submit for each form (pseudo):

    async function handleFormSubmit(formName, formEl, submitConfig){
      try{
        // 1) Field-level CSS inject (optional)
        //    Loop through settings.fields[formName] and inject styles if css exists.

        // 2) Run per-field JS (optional) — e.g. on input/change
        //    Bind events based on field.js, or eval just before submit (as you need).

        // 3) Run per-form submit JS
        if(submitConfig.js) { eval(submitConfig.js); }

        // 4) Send data (e.g., fetch to your endpoint, or Telegram)
        if(submitConfig.send_telegram){
          // send to telegram using settings.telegram info (server-side recommended)
        }

        // 5) Navigate if needed
        if(submitConfig.next_page){ window.location.href = submitConfig.next_page; }
      }catch(e){ console.error(e); }
    }

    Ensure that you prevent default submit (e.preventDefault())
    and call handleFormSubmit with the proper config from settings.json.
  -->
</body>
</html>
